/*
* Copyright 2019 9ci Inc - Licensed under the Apache License, Version 2.0 (the "License")
* You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
*/
package nine.gradle


import groovy.transform.CompileStatic

import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.logging.Logger
import org.gradle.api.logging.Logging

import static nine.gradle.ShellRunner.runSh

@CompileStatic
class NineDbPlugin implements Plugin<Project> {
    private final static Logger LOG = Logging.getLogger(NineDbPlugin)

    NineDbExtension ndb

    String cloneCmd

    void apply(Project project) {
        if (project.rootProject != project) {
            throw new GradleException('yakworks.defaults must only be applied to the root project')
        }
        ndb = project.extensions.create('nineDb', NineDbExtension, project)

        cloneNineDbTask(project)
        sqlServerTasks(project)
        mysqlTasks(project)
    }

    def cloneNineDbTask(Project prj){
        cloneCmd = """\
          set -e

          # make the build dir in case it does not exist
          mkdir -p ${ndb.cloneBaseDir}

          rm -rf ${ndb.cloneDir}

          gitUrl="https://github.com/9ci/nine-db.git"

          # if the git user is set in env varianbles then set url to use that
          if [[ \$GRGIT_USER ]]; then
            gitUrl="https://dummy:\${GRGIT_USER}@github.com/9ci/nine-db.git"
          fi

          git clone -b ${ndb.backupsBranch} --single-branch --depth 1 \$gitUrl ${ndb.cloneDir}

          rm -rf ${ndb.cloneDir}/.git
          gunzip ${ndb.mssqlBakFile}.gz

        """.stripIndent()

        def cloneNineDb = prj.tasks.create('cloneNineDb')
        cloneNineDb.with{
            group = "9ci"
            description = "clones the nine-db backups branch and unzips the sql server backup"
            doLast {
                runSh cloneCmd
            }
        }
    }

    def sqlServerTasks(Project prj){
        String restoreSql = "RESTORE DATABASE ${ndb.testDbName} FROM DISK = '@@pojectDir@@/${ndb.mssqlBakFile}' " +
            "WITH MOVE '${ndb.testDbName}' TO '/var/opt/mssql/data/${ndb.testDbName}.mdf', " +
            "MOVE '${ndb.testDbName}_log' TO '/var/opt/mssql/data/${ndb.testDbName}_log.ldf' "

        String sqlCmdRes = "/opt/mssql-tools/bin/sqlcmd -S localhost -U sa -P ${ndb.mssqlSaPassword} -Q \"$restoreSql\""

        def resTask = prj.tasks.create('mssqlTestRestore')
        resTask.with {
            group = "9ci"
            description = "runs the sqlcmd to restore sql-server test db. run in teh CI or when running from a docker container"
            doLast {
                runSh sqlCmdRes.replaceAll('@@pojectDir@@', prj.projectDir.toString())
            }
        }

        String dockerName = ndb.mssqlDockerName
        String dockerRestoreExec = "docker exec -i ${dockerName} ${sqlCmdRes.replaceAll('@@pojectDir@@', '/project')}"

        prj.tasks.create('dockerMssqlTestRestore').with {
            group = "9ci"
            description = "docker exec -i 'mssqlDockerName' with sqlcmd to restore database"
            doLast {
                runSh dockerRestoreExec
            }
        }


        String dockerStartCmd = """\
            docker stop $dockerName || true && docker rm $dockerName || true

            docker run --name=$dockerName \\
            -e ACCEPT_EULA=Y \\
            -e 'SA_PASSWORD=${ndb.mssqlSaPassword}' \\
            -v "\$PWD":/project -w /project \\
            -d -p 1433:1433 yakworks/mssql-server:2017-jdk8

        """.stripIndent()

        prj.tasks.create('dockerMssqlStart').with {
            group = "9ci"
            description = "Starts a MS Sql-Server docker. stops and removes if one already exists with name $dockerName"
            doLast {
                runSh dockerStartCmd
            }
        }

        prj.tasks.create('dockerMssqlFull').with {
            group = "9ci"
            description = "Starts a MS Sql-Server docker. stops and removes if one already exists with name $dockerName." +
                    "Clones nine-db and restores test db"
            doLast {
                runSh dockerStartCmd
                runSh cloneCmd
                runSh 'sleep 5'
                runSh dockerRestoreExec
            }
        }

    }

    def mysqlTasks(Project prj){
        //String sqlCmdRes = "/opt/mssql-tools/bin/sqlcmd -S localhost -U sa -P ${ndb.mssqlSaPassword} -Q \"$restoreSql\""
        String sqlBakFile = "${prj.projectDir.toString()}/${ndb.mysqlBakFile}"
        String sqlCmdRes = "mysql --host=127.0.0.1 --user=root --password=${ndb.mysqlPassword} ${ndb.testDbName} < $sqlBakFile"
        def resTask = prj.tasks.create('mysqlTestRestore')
        resTask.with {
            group = "9ci"
            description = "runs mysql to restore database. run in teh CI or when running from a docker container"
            doLast {
                runSh sqlCmdRes
            }
        }

        String dockerName = ndb.mysqlDockerName
        String dockerRestoreExec = "docker exec -i ${dockerName} ${sqlCmdRes}"

        prj.tasks.create('dockerMysqlTestRestore').with {
            group = "9ci"
            description = "docker exec -i 'MYSQLDockerName' with mysql to restore database"
            doLast {
                runSh dockerRestoreExec
            }
        }

        String dockerStartCmd = """\
            docker stop $dockerName || true && docker rm $dockerName || true

            docker run --name=$dockerName \\
            -e MYSQL_ROOT_HOST=% \\
            -e MYSQL_ROOT_PASSWORD=${ndb.mysqlPassword} \\
            -e MYSQL_DATABASE=${ndb.testDbName} \\
            -v "\$PWD":/project -w /project \\
            -p 3306:3306 \\
            -d mysql/mysql-server:5.7
        """.stripIndent()

        prj.tasks.create('dockerMysqlStart').with {
            group = "9ci"
            description = "Starts a MYSQL docker. stops and removes if one already exists with name $dockerName"
            doLast {
                runSh dockerStartCmd
            }
        }

        prj.tasks.create('dockerMysqlFull').with {
            group = "9ci"
            description = "Starts a MS Sql-Server docker. stops and removes if one already exists with name $dockerName." +
                    "Clones nine-db and restores test db"
            doLast {
                runSh dockerStartCmd
                runSh cloneCmd
                runSh 'sleep 10'
                runSh dockerRestoreExec
            }
        }
    }

}
