/*
* Copyright 2019 Yak.Works - Licensed under the Apache License, Version 2.0 (the "License")
* You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
*/
package gorm.tools.async

import groovy.transform.CompileStatic

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.transaction.TransactionStatus

import gorm.tools.transaction.WithTrx
import grails.persistence.support.PersistenceContextInterceptor

/**
 * a trait to be used for colating/slicing a list into "batches" to then asynchronously with Transactions
 * insert or update for maximum performance.
 *
 * @see GparsAsyncSupport  GparsAsyncSupport - for a concrete implementation
 *
 * @author Joshua Burnett (@basejump)
 * @since 6.1
 */
@CompileStatic
trait AsyncSupport implements WithTrx {
    final static Logger LOG = LoggerFactory.getLogger(AsyncSupport)

    /** The list size to send to the collate that slices.*/
    @Value('${hibernate.jdbc.batch_size:0}')
    int batchSize

    @Autowired
    PersistenceContextInterceptor persistenceInterceptor

    /**
     * calls {@link #parallel} with the args, batches list calling {@link #doBatch} for each batch in batches
     * passes itemClosure down through to the {@link #doBatch}
     *
     * @param args _optional_ arg map to be passed to the async engine such as gpars.
     *     can also add any other value and they will be passed down through the closure as well <br>
     *     - poolSize : gets passed down into the GParsPool.withPool for example
     * @param batches a collated list(batches) of sub-lists(a batch or items). each item in the batch(sub-list)
     *   will be asynchronously passed to the provided itemClosure. You should first collate the list with {@link #collate}
     * @param itemClosure the closure to call for each item lowest list.
     */
    void parallelBatch(Map args = [:], List<List> batches, Closure itemClosure) {
        parallel(args, batches) { List batch, Map cargs ->
            batchTrx(cargs, batch, itemClosure)
        }
    }

    /**
     * Iterates over the batchList of lists and calls the closure passing in the list and the args asynchronously.
     * parallelClosure it self is not asynchronous and will return once all the items in the batchList are processed.
     * If you want this method itself to be run asynchronous then it can be done in a Promise as outlined in the Grails async docs
     *
     * Must be overriden by the concrete implementation as this is where the meat is.
     *
     * @param args _optional_ arg map to be passed to the async engine such as gpars.
     *     can also add any other value and they will be passed down through the closure as well <br>
     *     - poolSize : gets passed down into the GParsPool.withPool for example
     * @param batches a collated list of lists. each batch list in the batches will be asynchronously passed to the provided closure
     * @param batchClosure the closure to call for each batch(sub-list of items) in the batches(list of batch sub-lists)
     */
    abstract void parallel(Map args = [:], List<List> batches, Closure batchClosure)

    /**
     * Iterates over the lists and calls the closure passing in the list item and the args asynchronously.
     * see GParsPoolUtil.eachParallel
     * Must be overriden by the concrete implementation as this is where the meat is.
     *
     * @param args _optional_ arg map to be passed to the async engine such as gpars.
     *     can also add any other value and they will be passed down through the closure as well <br>
     *     - poolSize : gets passed down into the GParsPool.withPool for example
     * @param batches a collated list of lists. each batch list in the batches will be asynchronously passed to the provided closure
     * @param itemClosure the closure to call for each item in list
     */
    abstract <T> Collection<T> eachParallel(Map args = [:], Collection<T> collection, Closure itemClosure)

    /**
     * Uses collate to break or slice the list into batches and then calls parallelBatch
     *
     * @param args optional arg map to be passed on through to eachParallel and the async engine such as gpars. <br>
     *     - batchSize : parameter to be passed into collate
     * @param list the list to process that will get sliced into batches via collate
     * @param itemClosure the closure to pass to eachParallel which is then passed to withTransaction
     */
    void parallelCollate(Map args = [:], List list, Closure itemClosure) {
        parallelBatch(args, collate(list, args.batchSize as Integer), itemClosure)
    }

    /**
     * Uses collate to break or slice the list into sub-lists of batches. Uses the {@link #batchSize} unless batchSize is sent in
     * @param items the list to slice up into batches
     * @param batchSize _optional_ the length of each batch sub-list in the returned list. override the {@link #batchSize}
     * @return the batches (list of lists) containing the chunked list of items
     */
    List<List> collate(List items, Integer batchSize = null) {
        items.collate(batchSize ?: getBatchSize()) as List<List>
    }

    /**
     * runs {@link #doBatch ) in a Transaction and flush and clear is called after doBatch but before commit.
     */
    void batchTrx(Map args, List items, Closure itemClosure) {
        withTrx { TransactionStatus status ->
            doBatch(args, items, itemClosure)
            //status.flush()
            //clear(status)
            flushAndClear(status)
        }

    }

    /**
     * calls closure for each item in list.
     *
     * @param args <i>optional args to pass to the closure. coming from parallelClosure
     * @param items the list or items to iterate over and run the closure
     * @param itemClosure the closure to execute for each item. will get passed the item and args as it itereates over the list
     */
    void doBatch(Map args, List items, Closure itemClosure) {
        for (Object item : items) {
            itemClosure.call(item, args)
        }
    }

    public <T> Closure<T> withSession(Closure<T> wrapped) {
        return { T item ->
            persistenceInterceptor.init()
            try {
                wrapped.call(item)
            } finally {
                try {
                    persistenceInterceptor.flush()
                    persistenceInterceptor.clear()
                } catch (Exception e) {
                    LOG.error("unexpected fail to flush and clear session after withSession", e);
                } finally {
                    try {
                        persistenceInterceptor.destroy()
                    } catch (Exception e) {
                        LOG.error("unexpected fail to flush and clear session after withSession", e);
                    }
                }
            }
        }
    }
}
